function [popt,resnorm,residual,ret]=Batchgauss2dfit(data,tilt,Use1dFit,Pos)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Fitting 2d gaussian function onto a 2d array of data ('data') using
    % the lsqcurvefit (Levenberg-Marquadt) algorithm in Matlab
    
    % Input: 
    % 'data' is the raw two dimensional data
    % 'grid' is an array consisting of the dimension in x (grid(1)) and y (grid(2)),
    % and the gridding in x and y (which will be placed right after the 
    % dimensions, in the same column array). This input is [nx ny x y].
    % 'tilt' is an optional input. If the value is one, fitting will be
    % done by considering the tilt of the 2d gaussian

    % Output:
    % The output consists of the fitted parameters ('popt'), the number of
    % iterations ('ret') and the initial parameters used ('pInit')
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Example
    % data = ones(100,100);               % Build data
    % dimension=size(data);               % Find out the dimensions
    % [ny,nx]=size(data);                 % Find out the dimensions
    % xAxis = 1:nx;                       % Create the x-Axis gridding
    % yAxis = 1:ny;                       % Create the y-Axis gridding
    % grid = [nx ny xAxis yAxis];         % Gridding input for gauss2dfunct and gauss2dfit
    % zOffset = 4.0;                      % Create parameters
    % Amplitude = 100.0;
    % xStd = 10.0;
    % yStd = 15.0;
    % xCenter = 45;
    % yCenter = 55;
    % tilt=0;                             % Value: 0 or 1. Specify whether or not there is tilt
    % tiltVal=0;                          % Value of tilt if tilt is to be
    % evaluated
    % p0=[zOffset, Amplitude, xStd, yStd, xCenter, yCenter,tiltVal];      % Combine parameters
    % z = gauss2dfunct(p0,grid);            % Find the gaussian function
    % z = reshape(z,ny,nx);             % Convert 1d array into 2d array
    % z = z+Amplitude/10*rand(ny,nx);   % Add some random noise
    % [popt,ret]=gauss2dfit(z,grid,tilt);     % Run gauss2dfit
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

    % Determine the dimensions of data
    [ny,nx,nz]=size(data);

    % Convert 2d array into 1d array
    z = reshape(data,ny*nx,1,nz);  

    % Find the initial estimate of the parameters
    % For 1d gaussian, the parameters refer to the constant term, amplitude 
    % term, stdev term and center term respectively.
    % For 2d circular gaussian: Parameters: p(1): z-offset, p(2): amplitude, 
    % p(3): xStdev, p(4): yStdev, p(5): xCenter, p(6): yCenter, p(7): tilt.

    % Initialize starting parameters for 2d Gaussian function
    p0 = zeros(nz,6);
    
    if Use1dFit == 1
        if isempty(Pos)
            [maxVal,~]=max(data,[],1);
            [~,maxX]=max(maxVal,[],2);
            [maxVal,~]=max(data,[],2);
            [~,maxY]=max(maxVal,[],1);
            Pos = [reshape(maxX,nz,1) reshape(maxY,nz,1)];
        end
            
        % Initialize dataXMat, dataMatY, p0x and p0y, which are parameters
        % for the 1d Gaussian function
        dataXMat = zeros(nz,nx);
        dataYMat = zeros(nz,ny);
        p0x = zeros(nz,4);
        p0y = zeros(nz,4);

        % Loop through all frames and get dataXMat, dataMatY, p0x and p0y
        for i = 1:nz
            dataTemp = data(:,:,i);

            % Get dataX, which is the pixels along x-axis passing through center
            yInd = round(Pos(i,2));
            yInd = yInd-1:yInd+1;
            yInd = yInd(yInd>0 & yInd<ny);
            dataX = dataTemp(yInd,:);
            dataX = mean(dataX,1);

            % Get dataY, which is the pixels along y-axis passing through center
            xInd = round(Pos(i,1));
            xInd = xInd-1:xInd+1;
            xInd = xInd(xInd>0 & xInd<nx);
            dataY = dataTemp(:,xInd);
            dataY = mean(dataY,2);

            % Smooth dataX and dataY
            xSmooth = smooth(dataX,3);  % Smoothing dataX with a span of 3
            ySmooth = smooth(dataY,3);  % Smoothing dataY with a span of 3

            % Finding bases
            [baseX,~] = min(xSmooth);     % Finding the base of smoothed data by finding the minimum value
            [baseY,~] = min(ySmooth);     % Finding the base of smoothed data by finding the minimum value

            % Getting offsets and amplitude
            yOffset = dataY - baseY;         % Substract ydata off the base to make the base zero
            [yAmp,~] = max(yOffset);     % Find center in y
            xOffset = dataX - baseX;         % Substract ydata off the base to make the base zero
            [xAmp,~] = max(xOffset);     % Find center in x

            % Find FWHM and Stdev in x and y
            yFWHM = length(find(yOffset>yAmp/2));
            yStdev = yFWHM/2.355;
            xFWHM = length(find(xOffset>xAmp/2));
            xStdev = xFWHM/2.355;

            % For 2d circular gaussian: Parameters: p(1): z-offset, p(2): amplitude, 
            % p(3): xStdev, p(4): yStdev, p(5): xCenter, p(6): yCenter, p(7): tilt.
            % p0(i,:) = [(baseX+baseY)/2 (xAmp+yAmp)/2 xStdev yStdev Pos(i,1) Pos(i,2)];

            p0x(i,:) = [baseX xAmp xStdev Pos(i,1)];
            p0y(i,:) = [baseY yAmp yStdev Pos(i,2)];
            dataXMat(i,:) = dataX;
            dataYMat(i,:) = dataY';
        end

        % Start 1d Gaussian optimization options
        options = optimset('Display','off',...
        'Algorithm','levenberg-marquardt','Jacobian','on','TolFun',1e-6,'TolX',1e-6);
    
        % Loop through all frames and do 1d Gaussian curve fitting
        px = zeros(nz,4);
        py = zeros(nz,4);
        for i = 1:nz
            try
                [px(i,:),~,~,~,~] = lsqcurvefit(@gauss1dfunct,p0x(i,:),1:nx,dataXMat(i,:),[],[],options);
                [py(i,:),~,~,~,~] = lsqcurvefit(@gauss1dfunct,p0y(i,:),1:ny,dataYMat(i,:),[],[],options);
            catch
                px(i,:) = [0 1 1 1];
                py(i,:) = [0 1 1 1];
            end
        end

        % Estimate p0 from px and py
        p0 = [((px(:,1)+py(:,1))/2) sqrt(abs(px(:,2).*py(:,2))) px(:,3) py(:,3) px(:,4) py(:,4)];
    else
        if isempty(Pos)
            [maxVal,~]=max(data,[],1);
            [~,maxX]=max(maxVal,[],2);
            [maxVal,~]=max(data,[],2);
            [~,maxY]=max(maxVal,[],1);
            Pos = [reshape(maxX,nz,1) reshape(maxY,nz,1)];
        end
        
        % Loop through all frames and get p0
        for i = 1:nz
            dataTemp = data(:,:,i);

            % Get dataX, which is the pixels along x-axis passing through center
            yInd = round(Pos(i,2));
            yInd = yInd-1:yInd+1;
            yInd = yInd(yInd>0 & yInd<ny);
            dataX = dataTemp(yInd,:);
            dataX = mean(dataX,1);

            % Get dataY, which is the pixels along y-axis passing through center
            xInd = round(Pos(i,1));
            xInd = xInd-1:xInd+1;
            xInd = xInd(xInd>0 & xInd<nx);
            dataY = dataTemp(:,xInd);
            dataY = mean(dataY,2);

            % Smooth dataX and dataY
            xSmooth = smooth(dataX,3);  % Smoothing dataX with a span of 3
            ySmooth = smooth(dataY,3);  % Smoothing dataY with a span of 3

            % Finding bases
            [baseX,~] = min(xSmooth);     % Finding the base of smoothed data by finding the minimum value
            [baseY,~] = min(ySmooth);     % Finding the base of smoothed data by finding the minimum value

            % Getting offsets and amplitude
            yOffset = dataY - baseY;         % Substract ydata off the base to make the base zero
            [yAmp,~] = max(yOffset);     % Find center in y
            xOffset = dataX - baseX;         % Substract ydata off the base to make the base zero
            [xAmp,~] = max(xOffset);     % Find center in x

            % Find FWHM and Stdev in x and y
            yFWHM = length(find(yOffset>yAmp/2));
            yStdev = yFWHM/2.355;
            xFWHM = length(find(xOffset>xAmp/2));
            xStdev = xFWHM/2.355;

            % For 2d circular gaussian: Parameters: p(1): z-offset, p(2): amplitude, 
            % p(3): xStdev, p(4): yStdev, p(5): xCenter, p(6): yCenter, p(7): tilt.
            p0(i,:) = [(baseX+baseY)/2 (xAmp+yAmp)/2 xStdev yStdev Pos(i,1) Pos(i,2)];
        end
    end

    % If tilt is specified, simply initialize the tilt with zero
    if tilt==1
        p0 = [p0 zeros(nz,1)];
    end
    
    % Define options for fitting
    options = optimset('Display','off',...
    'Algorithm','levenberg-marquardt','Jacobian','on','TolFun',1e-10,'TolX',1e-10);
    grid = [nx ny 1:nx 1:ny];    
    
    % Initialize output
    popt = zeros(nz,8);
    resnorm = zeros(nz,1);
    residual = cell(nz,1);
    ret = zeros(nz,1);

    % For popt: popt(1): z-offset, popt(2): amplitude, popt(3): xStdev, popt(4): yStdev, 
    % popt(5): xCenter, popt(6): yCenter, popt(7): tilt, popt(8): error
    % status, popt(9): time
    % Error status: 0 if no error, 1 if center is outside image, 2 if there
    % is fitting error

    for i = 1:nz
        try
            [popt(i,1:size(p0,2)),resnorm(i),residual{i},~,output] = lsqcurvefit(@gauss2dfunct,p0(i,:),grid,z(:,:,i),[],[],options);
            ret(i)=output.iterations;
            if ~(popt(i,5)>0 && popt(i,5)<nx && popt(i,6)>0 && popt(i,6)<ny)
                popt(i,1:size(p0,2)) = zeros(1,size(p0,2));
                popt(i,8) = 1;
            end
        catch
            popt(i,1:size(p0,2)) = zeros(1,size(p0,2));
            popt(i,8) = 2;
        end
    end
end
